﻿/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
 © i-net software 1998-2013
*/  
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using System.Linq;

using Inet.Viewer.Data;
using Inet.Viewer.Resources;
using Inet.Viewer.WinForms.Export;

namespace Inet.Viewer.WinForms
{
    /// <summary>
    /// An implemenation of the interface <see cref="IToolBar"/>. This Toolbar changes its behaviour according to the 
    /// currently added <see cref="ReportView"/>.
    /// </summary>
    [ToolboxItem(false)] 
    public partial class ViewerToolbar : UserControl, IToolBar
    {

        private const string ZoomFitToPage = "fit to page";
        private const string ZoomFitToWidth = "fit to width";
        private const string ZoomFitToHeight = "fit to height";

        /// <summary>
        /// int array that holds the different zoom levels that are shown
        /// </summary>
        private static readonly int[] ZoomStep = new int[] { 10, 25, 50, 75, 100, 150, 200, 400, };

        /// <summary>
        /// int array that holds the different zoom levels that are used for Zoom + and - feature
        /// </summary>        
        private static readonly int[] ZoomStepsDetail = new int[] { 10, 18, 25, 33, 50, 66, 75, 88, 100, 125, 150, 175, 200, 300, 400, 600 };

        /// <summary>
        /// Used for the Property ReportView
        /// </summary>
        private ReportView reportView;

        private DataChanged dataInitedHandler;
        private PageChanged pageChangedHandler;
        private EventHandler zoomChangedHandler;
        /// <summary>
        /// Constructor sets the zoom steps and disables all buttons by default
        /// </summary>
        public ViewerToolbar()
        {
            InitializeComponent();
            
            dataInitedHandler = new DataChanged((x, y) => {
                if (InvokeRequired)
                {
                    Invoke(new Action(UpdateButtonStates));
                }
                else
                {
                    UpdateButtonStates();
                }
            });
            pageChangedHandler = new PageChanged(ReportViewPageChanged);
            zoomChangedHandler = new EventHandler(ViewZoomChanged);

            UpdateButtonStates();            
            SetupZoomLevel();
            SetLabels();
        }
 
        /// <summary>
        /// Sets the visiblity of the buttons in the toolbar, depending on the availability 
        /// of the representing functions
        /// </summary>
        private void UpdateButtonStates()
        {
            btnRefresh.Enabled = reportView != null;

            bool state = reportView != null && reportView.ReportInfo != null;
            toolStripSeparatorRefresh.Visible = state;

            btnExport.Enabled = state;

            btnPrint.Enabled = state && reportView.ReportInfo.IsPrintingEnabled;

            btnZoomLevel.Enabled = state;

            btnZoomMinus.Enabled = state;

            btnZoomPlus.Enabled = state;

            btnSearch.Enabled = state;
            btnTool.Enabled = state;
            btnPageSingle.Enabled = state;
            btnPageSingleCont.Enabled = state;
            btnPageDouble.Enabled = state;
            btnPageDoubleCont.Enabled = state;

            DisplayNavigationButtons();
        }

        /// <summary>
        /// Updates the label for PageCount and CurrentPage
        /// </summary>
        private void UpdatePageLabelAndText()
        {
            if (ReportView != null)
            {
                this.txtCurrentPage.Enabled = true;
                string pageLimit = string.Empty;
                if (ReportView.PageLimitExceeded)
                {
                    pageLimit = "+";
                }
                string pageCount = ReportView.PageCount < 1 ? "?" : ReportView.PageCount + string.Empty;
                this.txtCurrentPage.Text = ReportView.CurrentPage + " / " + pageCount + pageLimit;
                this.Refresh();
            }
            else
            {
                // no report shown
                this.txtCurrentPage.Enabled = false;
                this.txtCurrentPage.Text = string.Empty;
            }
        }
     
        /// <summary>
        /// This methods enables or disables the navigation button according to the <see cref="ReportView"/>.CurrentPage of the ReportView.    
        /// The <see cref="ReportView"/> needs to be set for this.
        /// </summary>
        private void DisplayNavigationButtons()
        {
            // Disable Navigation Buttons
            btnPreviousPage.Enabled = false;
            btnLastPage.Enabled = false;
            btnNextPage.Enabled = false;
            btnFirstPage.Enabled = false;

            txtCurrentPage.Enabled = false;
            txtCurrentPage.Enabled = false;

            if (ReportView != null && ReportView.ReportInfo != null)
            {
                bool isFirstPage = ReportView.CurrentPage == 1;
                // If last page is unknown it stil has the Default value 0
                bool isLastPage = ReportView.CurrentPage == ReportView.PageCount;
                bool isInBetweenPage = !isFirstPage && !isLastPage;
                bool hasOnlyOnePage = ReportView.PageCount == 1;

                txtCurrentPage.Enabled = true;
                if ((isFirstPage || isInBetweenPage) && !hasOnlyOnePage)
                {
                    btnNextPage.Enabled = true;
                    btnLastPage.Enabled = ReportView.PageCount > 0;
                }

                if ((isLastPage || isInBetweenPage) && !hasOnlyOnePage)
                {
                    btnPreviousPage.Enabled = true;
                    btnFirstPage.Enabled = true;
                }
            }
        }

        /// <summary>
        /// Updates the labels and the Navigation buttons
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="page">the page</param>
        private void ReportViewPageChanged(object sender, int page)
        {
            UpdatePageLabelAndText();
            DisplayNavigationButtons();
        }

        /// <summary>
        /// Update CurrentPage when the 'Enter' key is pressed in the CurrentPage textfield
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the arguments</param>
        private void txtCurrentPage_KeyUp(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Return && ReportView != null)
            {
                string page = txtCurrentPage.Text;
                // if has page separator
                if (page != null && page.Contains("/"))
                {
                    int index = page.IndexOf("/");
                    page = page.Substring(0, index);
                }

                int pageNumber;
                if (int.TryParse(page, out pageNumber))
                {
                    ReportView.CurrentPage = pageNumber;
                }
                UpdatePageLabelAndText();
            }
        }
      
        /// <summary>
        /// Next Page Button clicked
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the arguments</param>
        private void btnNextPage_Click(object sender, System.EventArgs e)
        {
            if (this.ReportView != null)
            {
                ReportView.NextPage();
                UpdatePageLabelAndText();                
            }
        }

        /// <summary>
        /// Print button clicked
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the arguments</param>
        private void btnPrint_Click(object sender, System.EventArgs e)
        {
            // open the dialog async due to an issue with the printer dialog.
            // otherwise the dialog doesn't get the input focus when opened from toolstrip click event.
            BeginInvoke((MethodInvoker)delegate {
                ReportView.Print();
            });
        }

        /// <summary>
        /// Last Page button clicked
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the arguments</param>
        private void btnLastPage_Click(object sender, System.EventArgs e)
        {
            if (this.ReportView != null)
            {
                ReportView.LastPage();
            }
        }

        /// <summary>
        /// Previous Page button clicked
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the arguments</param>
        private void btnPreviousPage_Click(object sender, System.EventArgs e)
        {
            if (this.ReportView != null)
            {
                ReportView.PreviousPage();
            }
        }

        /// <summary>
        /// First Page button clicked
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the arguments</param>
        private void btnFirstPage_Click(object sender, System.EventArgs e)
        {
            if (this.ReportView != null)
            {
                ReportView.FirstPage();
            }
        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>    
        public void SetButtonsVisible(ToolbarButtonType buttonType, bool visible)
        {
            switch (buttonType)
            {
                case ToolbarButtonType.General:                        
                    break;
                case ToolbarButtonType.Navigation:
                    break;          
            }
        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>    
        public bool IsButtonsVisible(ToolbarButtonType buttonType)
        {
            switch (buttonType)
            {
                case ToolbarButtonType.General:

                    break;
                case ToolbarButtonType.Navigation:

                    break;

            }
            return false;
        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        public IReportView ReportView
        {
            get
            {
                return reportView;
            }
            set
            {
                // remove events
                if (reportView != null)
                {
                    // remove events from previous ReportView
                    reportView.DataInited -= dataInitedHandler;
                    reportView.PageChanged -= pageChangedHandler;
                    reportView.ZoomChanged -= zoomChangedHandler;                
                }
                
                reportView = (ReportView)value;                
                // setup up the toolbar
                if (reportView != null)
                {                    
                    UpdatePageLabelAndText();
                    value.DataInited += dataInitedHandler;
                    value.PageChanged += pageChangedHandler;
                    this.ShowZoomLevel();
                    value.ZoomChanged += zoomChangedHandler;
                    this.ShowMouseMode();
                    switch(reportView.PageViewMode) {
                        case PageViewMode.SinglePage:
                            MarkPageViewModeButtonChecked(btnPageSingle);
                            break;
                        case PageViewMode.SingleContinuous:
                            MarkPageViewModeButtonChecked(btnPageSingleCont);
                            break;
                        case PageViewMode.DoublePage:
                            MarkPageViewModeButtonChecked(btnPageDouble);
                            break;
                        case PageViewMode.DoubleContinuous:
                            MarkPageViewModeButtonChecked(btnPageDoubleCont);
                            break;
                    }
                }
                UpdateButtonStates();
            }
        }

        /// <summary>
        /// Called when the zoom level was changed.
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the arguments</param>
        private void ViewZoomChanged(object sender, EventArgs e)
        {
            ShowZoomLevel();
        }
 
        /// <summary>
        /// Adds the zoom levels to the Zoom level combobox
        /// </summary>
        private void SetupZoomLevel()
        {
            for (int i = 0; i < ZoomStep.Length; i++)
            {
                btnZoomLevel.Items.Add(ZoomStep[i] + " %");
            }                          
            btnZoomLevel.Items.Add(strings.FitToPage);
            btnZoomLevel.Items.Add(strings.FitToHeight);
            btnZoomLevel.Items.Add(strings.FitToWidth);

        }

        /// <summary>
        /// Decreases the zoom level
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the arguments</param>
        private void btnZoomMinus_Click(object sender, EventArgs e)
        {
            ZoomOut();
        }

        /// <summary>
        /// Increases the zoom level
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the arguments</param>
        private void btnZoomPlus_Click(object sender, EventArgs e)
        {
             ZoomIn();
        }    

        /// <summary>
        /// This method calculates the next smallest step of the zoom level.
        /// Uses the ZOOM_STEPS array for the information. With inbetween steps.
        /// Implementation for float values
        /// </summary>
        /// <param name="currentZoom">The zoom value in percent in float (1.0 = 100%)</param>
        /// <returns>zoom float value in percent (1.0 = 100%)</returns>
        public static float NextHigherZoomLevel(float currentZoom)
        {
            float lastStep = 0;
            foreach (int step in ZoomStepsDetail)
            {
                float fStep = (float) step / 100f;
                if (currentZoom < fStep) 
                {
                    return fStep;
                }
                lastStep = step;
            }
            return lastStep;
        }
     
        /// <summary>
        /// This method calculates the next bigger step of the zoom level
        /// Uses the ZOOM_STEPS array for the information        
        /// </summary>
        /// <param name="currentZoom">The zoom value in percent</param>
        /// <returns>zoom float value in percent (1.0 = 100%)</returns>
        public static float NextLowerZoomLevel(float currentZoom)
        {
            float smallestStep = ZoomStepsDetail[0] / 100f;
            for (int i = ZoomStepsDetail.Length - 1; i > 0; i--)
            {
                float step = (float) ZoomStepsDetail[i] / 100f;
                if (currentZoom > step)
                {
                    return step;
                }
            }
            return smallestStep;
        }

        /// <summary>
        /// This method displays the current Zoom factor in the ZoomLevel Combobox
        /// </summary>
        private void ShowZoomLevel()
        {
            if (ReportView != null)
            {                
                switch (ReportView.ZoomMode)
                {
                    case ZoomMode.Manual:
                        this.btnZoomLevel.Text = ReportView.ZoomFactor * 100 + " %";
                        break;
                    case ZoomMode.FullPage:
                        this.btnZoomLevel.Text = strings.FitToPage;                       
                        break;

                    case ZoomMode.PageHeight:
                        this.btnZoomLevel.Text = strings.FitToHeight;                       
                        break;

                    case ZoomMode.PageWidth:
                        this.btnZoomLevel.Text = strings.FitToWidth;                      
                        break;
                }                
            }
        }

        /// <summary>
        /// Updates the shown mouse mode with the current mouse
        /// mode from the report view.
        /// </summary>
        private void ShowMouseMode()
        {
            if (ReportView != null)
            {
                switch (ReportView.MouseActionMode)
                {
                    case MouseMode.Panning:
                        this.btnTool.Image = global::Inet.Viewer.Images.hand;
                        break;
                    case MouseMode.SelectText:
                        this.btnTool.Image = global::Inet.Viewer.Images.textselect;
                        break;
                }
            }
        }

        /// <summary>
        /// Parses the Export format that is save in the Tag of the ToolStripDropDownItem
        /// </summary>
        /// <param name="sender"></param>
        /// <returns>Returns the parsed export format</returns>
        private static ExportFormat GetExportFormat(object sender)
        {
            ExportFormat selectedFormat = ExportFormat.None;
            // Get the export format
            ToolStripDropDownItem item = sender as ToolStripDropDownItem;
            int intExportFormat = -1;
            if (item != null && item.Tag != null && int.TryParse(item.Tag.ToString(), out intExportFormat))
            {
                selectedFormat = (ExportFormat)intExportFormat;
            }
            return selectedFormat;
        }
      
        /// <summary>
        ///  Does the localisation of all the strings with the current CultureInfo
        /// </summary>
        private void SetLabels()
        {
            btnFirstPage.ToolTipText = strings.FirstPage;
            btnLastPage.ToolTipText = strings.LastPage;
            btnPreviousPage.ToolTipText = strings.PreviousPage;
            btnNextPage.ToolTipText = strings.NextPage;
            txtCurrentPage.ToolTipText = strings.CurrentPage;
            btnZoomLevel.ToolTipText = strings.ZoomLevel;
        }

        /// <summary>
        /// Adds a progress to the toolbar.
        /// </summary>
        /// <param name="progress">the progress to add</param>
        public void AddProgress(Progress progress) 
        {
            ToolStripProgress toolStripProgress = new ToolStripProgress(progress);
            toolStripProgress.Finished += ToolStripProgressFinished;
            toolStrip1.Items.Add(toolStripProgress);
        }

        /// <inheritdoc/>
        public IEnumerable<Progress> SelectProgress(Func<Progress, bool> filter)
        {
            return toolStrip1.Items.Cast<ToolStripItem>().Where(e => e is ToolStripProgress && filter(((ToolStripProgress)e).Progress)).Select(e => ((ToolStripProgress)e).Progress);
        }

        /// <summary>
        /// The checked/unchecked state of the search button.
        /// </summary>
        public bool SearchChecked
        {
            set { btnSearch.Checked = value; }
        }

        /// <summary>
        /// Implements the setting of the Zoom with the Combobox btnZoomLevel
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the arguments</param>
        private void btnZoomLevel_SelectedIndexChanged(object sender, EventArgs e)
        {
            object item = btnZoomLevel.Items[btnZoomLevel.SelectedIndex];
            string sZoom = item as string;
            if (sZoom != null && ReportView != null)
            {
                // is a percent value
                if (sZoom.Contains("%"))
                {
                    int index = sZoom.IndexOf("%");
                    sZoom = sZoom.Substring(0, index);
                    int iZoom = 0;
                    if (int.TryParse(sZoom, out iZoom))
                    {
                        ReportView.ZoomMode = ZoomMode.Manual;
                        ReportView.ZoomFactor = (float)iZoom / 100;
                        ShowZoomLevel();
                    }
                }
                else
                {
                    // is a string
                    if (sZoom.Equals(strings.FitToPage))
                    {
                        this.ReportView.ZoomMode = ZoomMode.FullPage;
                    }
                    else if (sZoom.Equals(strings.FitToWidth))
                    {
                        this.ReportView.ZoomMode = ZoomMode.PageWidth;
                    }
                    else if (sZoom.Equals(strings.FitToHeight))
                    {
                        this.ReportView.ZoomMode = ZoomMode.PageHeight;
                    }
                }
            }
        }

        /// <summary>
        /// This method sets the Zoom after a valid zoom number was entered in the comobobox btnZoomLevel
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the arguments</param>
        private void btnZoomLevel_KeyUp(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Return)
            {
                string sZoom = btnZoomLevel.Text;
                if (sZoom != null)
                {
                    if (sZoom.Contains("%"))
                    {
                        int index = sZoom.IndexOf("%");
                        sZoom = sZoom.Substring(0, index);
                    }
                    int iZoom = 0;
                    if (int.TryParse(sZoom, out iZoom))
                    {
                        if (ReportView != null)
                        {
                            ReportView.ZoomFactor = (float)iZoom / 100;
                        }
                    }
                    ShowZoomLevel();
                }
            }
        }

        /// <summary>
        /// Called when the refresh button was clicked.
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the arguments</param>
        private void btnRefresh_Click(object sender, EventArgs e)
        {
            if (ReportView != null)
            {
                ReportView.DataRefresh();
                // for the case the page count changed
                UpdatePageLabelAndText();
                DisplayNavigationButtons();
            }
        }       
      
        /// <summary>
        /// Callback wich is called when a progress is finished. Removes the progress toolbar item.
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the arguments</param>
        private void ToolStripProgressFinished(object sender, EventArgs e)
        {
            if (toolStrip1.InvokeRequired)
            {
                // non-UI threads may send this event
                toolStrip1.Invoke(new Action<object, EventArgs>(ToolStripProgressFinished), new [] {sender, e});
                return;
            }
            try
            {
                toolStrip1.Items.Remove((ToolStripProgress)sender);
            }
            catch (Exception ex)
            {
                // ignore
                ViewerUtils.PrintStackTrace(ex);
            }
        }
 
        /// <summary>
        /// Called when the search button was clicked.
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the event arguments</param>
        private void btnSearch_Click(object sender, EventArgs e)
        {
            reportView.ReportViewer.SearchVisible =  btnSearch.Checked;
        }

        /// <summary>
        /// Called when a tool drop-down item was clicked.
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the event arguments</param>
        private void btnToolDropDown_ItemClicked(object sender, ToolStripItemClickedEventArgs e)
        {
            btnTool.Image = e.ClickedItem.Image;
            reportView.MouseActionMode = (MouseMode)Enum.Parse(typeof(MouseMode), (string)e.ClickedItem.Tag);
        }

        /// <summary>
        /// Called when the export button was clicked.
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the event arguments</param>
        private void btnExport_Click(object sender, EventArgs e)
        {
            reportView.OpenExportDialog();
        }

        /// <summary>
        /// Called when the user clicks on a page view mode button.
        /// </summary>
        /// <param name="sender">the sender</param>
        /// <param name="e">the event arguments</param>
        private void btnPageMode_Click(object sender, EventArgs e)
        {
            reportView.PageViewMode = (PageViewMode)Enum.Parse(typeof(PageViewMode), (string)((ToolStripButton)sender).Tag);
            MarkPageViewModeButtonChecked((ToolStripButton)sender);
        }

        /// <summary>
        /// Mark the the specified button checked and all other page view mode buttons unchecked.
        /// </summary>
        /// <param name="sender">the button to check</param>
        private void MarkPageViewModeButtonChecked(ToolStripButton sender)
        {
            btnPageSingle.Checked = sender == btnPageSingle;
            btnPageSingleCont.Checked = sender == btnPageSingleCont;
            btnPageDouble.Checked = sender == btnPageDouble;
            btnPageDoubleCont.Checked = sender == btnPageDoubleCont;
        }

        private void btnZoomLevel_DropDownClosed(object sender, EventArgs e)
        {
            ReportView.Focus();
        }

        /// <inheritdoc/>
        public void ZoomIn()
        {
            if (ReportView != null)
            {
                ReportView.ZoomMode = ZoomMode.Manual;
                ReportView.ZoomFactor = NextHigherZoomLevel(ReportView.ZoomFactor);
                ShowZoomLevel();
            }
        }

        /// <inheritdoc/>
        public void ZoomOut()
        {
            if (ReportView != null)
            {
                ReportView.ZoomMode = ZoomMode.Manual;
                ReportView.ZoomFactor = NextLowerZoomLevel(ReportView.ZoomFactor);
                ShowZoomLevel();
            }
        }

    }
}

